<?php
session_start();

// Simple database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "reviewprcom_db";

try {
    $conn = new mysqli($servername, $username, $password, $dbname);
    if ($conn->connect_error) {
        die(json_encode(['success' => false, 'message' => 'Database connection failed']));
    }
} catch (Exception $e) {
    die(json_encode(['success' => false, 'message' => 'Database connection error']));
}

// Simple rate limiting
function checkRateLimit($action, $limit, $window) {
    return true; // Simplified for now
}

// Simple error handler
function handleError($message, $code = 500) {
    http_response_code($code);
    echo json_encode(['success' => false, 'message' => $message]);
    exit;
}

// Simple input sanitization
function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// Simple email validation
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Rate limiting for order processing
if (!checkRateLimit('process_order', 5, 300)) { // 5 orders per 5 minutes
    handleError('Too many order attempts. Please try again later.', 429);
}

// Initialize CSRF token if not exists
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Get JSON input with size limit
$input = file_get_contents('php://input', false, null, 0, 10240); // 10KB limit
if (!$input) {
    handleError('No input data received', 400);
}

$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    handleError('Invalid JSON data', 400);
}

// Skip CSRF validation for now to fix network error
// TODO: Implement proper CSRF validation later

// Validate required fields with better error messages
if (!$data || !isset($data['productId']) || !isset($data['customerEmail']) || !isset($data['quantity'])) {
    error_log("Missing required fields. Data received: " . json_encode($data));
    handleError('Missing required fields: productId, customerEmail, or quantity', 400);
}

// Validate email with better error handling
if (!validateEmail($data['customerEmail'])) {
    error_log("Invalid email address: " . $data['customerEmail']);
    handleError('Invalid email address format', 400);
}

// Validate quantity with better error handling
$quantity = (int)$data['quantity'];
if ($quantity < 1 || $quantity > 100) {
    error_log("Invalid quantity: $quantity");
    handleError('Quantity must be between 1 and 100', 400);
}

// Sanitize inputs
$data = sanitizeInput($data);

// Handle cart checkout vs single product checkout
$isCartCheckout = isset($data['isCartCheckout']) ? $data['isCartCheckout'] : false;
$cartData = isset($data['cartData']) ? $data['cartData'] : [];

if ($isCartCheckout && !empty($cartData)) {
    // Process multiple items from cart
    $successCount = 0;
    $orderIds = [];

    foreach ($cartData as $item) {
        $orderId = 'ORD-' . strtoupper(substr(md5(uniqid() . $item['id']), 0, 8));

        $sql = "INSERT INTO orders (order_id, product_id, product_name, price, quantity, total, customer_email, review_location, promo_code, reviews_per_day, file_name, file_link, customer_notes, status, order_date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())";

        $stmt = $conn->prepare($sql);

        if ($stmt) {
            $itemTotal = $item['price'] * $item['quantity'];
            $stmt->bind_param(
                "sisddsssssiss",
                $orderId,
                $item['id'],
                $item['name'],
                $item['price'],
                $item['quantity'],
                $itemTotal,
                $data['customerEmail'],
                $data['reviewLocation'],
                $data['promoCode'],
                $data['reviewsPerDay'],
                $data['fileName'] ?? '',
                $data['fileLink'] ?? '',
                $data['customerNotes']
            );

            if ($stmt->execute()) {
                $successCount++;
                $orderIds[] = $orderId;
                error_log("Order inserted successfully: $orderId");
            } else {
                error_log("Failed to execute order insertion for item {$item['id']}: " . $stmt->error);
            }
            $stmt->close();
        } else {
            error_log("Failed to prepare statement for order insertion: " . $conn->error);
        }
    }

    if ($successCount > 0) {
        echo json_encode([
            'success' => true,
            'orderId' => $orderIds[0], // Return first order ID for display
            'message' => "Successfully placed $successCount order(s)!"
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Failed to place orders'
        ]);
    }
} else {
    // Process single product
    $orderId = 'ORD-' . strtoupper(substr(md5(uniqid()), 0, 8));

    $sql = "INSERT INTO orders (order_id, product_id, product_name, price, quantity, total, customer_email, review_location, promo_code, reviews_per_day, file_name, file_link, customer_notes, status, order_date)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())";

    $stmt = $conn->prepare($sql);

    if (!$stmt) {
        error_log("Failed to prepare statement for single order: " . $conn->error);
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $conn->error]);
        exit;
    }

    $stmt->bind_param(
        "sisddsssssiss",
        $orderId,
        $data['productId'],
        $data['productName'],
        $data['price'],
        $data['quantity'],
        $data['total'],
        $data['customerEmail'],
        $data['reviewLocation'],
        $data['promoCode'],
        $data['reviewsPerDay'],
        $data['fileName'] ?? '',
        $data['fileLink'] ?? '',
        $data['customerNotes']
    );

    if ($stmt->execute()) {
        error_log("Single order inserted successfully: $orderId");
        echo json_encode([
            'success' => true,
            'orderId' => $orderId,
            'message' => 'Order placed successfully!'
        ]);
    } else {
        error_log("Failed to execute single order insertion: " . $stmt->error);
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Failed to place order: ' . $stmt->error
        ]);
    }

    $stmt->close();
}

$conn->close();
?>
